
//	=================================================================
//	ADDON.C						31-01-2011 Arttu Soininen
//
//	Example addon application for TerraScan.
//
//	Compiler: Microsoft Visual C++ 6.0
//	Required files:
//	 - addon.mke		make file (used with bmakewin)
//	 - addon.c			this source module
//	 - addon.h			structure definitions
//	 - resource.h		resource identifiers
//	 - addon.res		resource
//	 - addon.def		DLL definition
//	=================================================================

#define WINVER	0x0400

#include <windef.h>
#include <windows.h>
#include <windowsx.h>
#include <commdlg.h>
#include <shellapi.h>

#include <winuser.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#include "addon.h"
#include "resource.h"

#define WINDLG		LRESULT CALLBACK

static HINSTANCE	DllInst ;			//	Module instance

//	Pointer to TerraScan variables

static AddLnk		*LnkPtr = NULL ;	//	Link to TerraScan variables

//	View histogram variables

static HWND			HisDlg = NULL ;
static int			HisClass = CLASS_ANY ;
static int			HisScanner = SCANNER_ANY ;
static int			*HisTbl = 0 ;		//	Histogram values
static int			HisHgt = 0 ;		//	Picture height in pixels
static int			HisMax = 0 ;		//	Maximum intensity
static int			HisPeak = 0 ;		//	Biggest count in histogram
static double		HisAve = 0.0 ;		//	Average intensity
static double		HisMedian = 0 ;		//	Median intensity
static double		HisSpread = 0 ;		//	Spread around median for 80% of points
static const int	HisMargY = 10 ;		//	Y axis margin
static const int	HisMargX = 2 ;		//	X axis margin

//	Internal prototypes

int		ViewHistogramCmd( AddSet *Sp) ;
WINDLG	HistogramDlg( HWND Dlg, UINT Msg, WPARAM Par1, LPARAM Par2) ;
void	HisDrawPicture( HDC Hdc, RECT *Rt) ;
void	HisDrawLabel( HDC Hdc, int Val, int Max, RECT *Tb) ;
int		HisLabelStep( int Max) ;
void	HisRecompute( HWND Dlg, int Redraw) ;
int		HisCompute( int *Peak, int *Tbl, USHORT *Int, BYTE *Mrk, BYTE *Cls, BYTE *Scr, int Pct, int Class, int Scv) ;
int		HisCompAveMedian( double *Ave, double *Med, double *Spr, int *Tbl, int Cnt) ;

void	ClassFillDrop( HWND Drop, PntCls *Tbl, int Cnt, int SelVal) ;
int		ClassGetDropValue( HWND Drop, PntCls *Tbl, int Cnt) ;

int		ScannerGetUsed( BYTE *Sct, BYTE *Scr, int Cnt) ;
void	ScannerFillDrop( HWND Drop, BYTE *Sct, int SelVal) ;
int		ScannerGetDropValue( HWND Drop, BYTE *Scr) ;

int		RunMacroStep( char *Cmd, char *Unparsed, char *BlkFile, char *ActFile) ;
int		ScanAssignEcho( BYTE *Ech, Point3d *Pnt, BYTE *Mrk, double *Dbl, int Cnt) ;
int		ScanMultiplyIntensity( USHORT *Int, int Cnt, double Mul) ;

void	ScanMarkAll( BYTE *Mrk, int Cnt, int Mark) ;
int		ScanMarkClass( BYTE *Mrk, BYTE *Cls, int Cnt, int Class, int Mark) ;
int		ScanMarkScanner( BYTE *Mrk, BYTE *Scr, int Cnt, int Scv, int From, int To) ;
int		ScanIntensityRange( USHORT *Mn, USHORT *Mx, USHORT *Int, BYTE *Mrk, int Cnt, int Mark) ;

void	DpToIp( Point3d *P, Dp3d *D, AddLnk *Lp) ;
void	IpToDp( Dp3d *D, Point3d *P, AddLnk *Lp) ;
long	DblToLong( double D) ;

//	=================================================================
//	DLL entry function which is called when TSCAN is loaded.
//	
//	Addon application should store module instance Inst.
//	=================================================================

int  __stdcall	LibMain( HINSTANCE Inst, DWORD Reason, void *Reserved)
{
	DllInst = Inst ;

	return (1) ;
}

//	=================================================================
//	Required by MicroStation.
//	=================================================================

int		initialize( void)
{
	return (0) ;
}

//	=================================================================
//	Required for running DLL under PowerMap or PowerCivil.
//	=================================================================

extern void		AddonSetPassword( char *Pwd, char *App) ;

void	dlmPassword( char *Pwd, char *App)
{
	AddonSetPassword( Pwd, App) ;
}

//	=================================================================
//	TerraScan calls this routine at startup so that addon application
//	can fill Tbl[] with information about command names that should
//	appear in the pulldown menu.
//
//	TerraScan has allocated Tbl for upto MaxCnt items.
//
//	Addon can also initialize variables and allocate resources.
//
//	Return value must specify number of items filled into Tbl[].
//	=================================================================

int		AddonFillCommands( AddCmd *Tbl, int MaxCnt)
{
	//	Validate parameters

	if ((!Tbl) || (MaxCnt < 1))					return (0) ;

	//	Store information about 'View histogram' tool

	strcpy( Tbl[0].MenuText, "View histogram...") ;
	Tbl[0].UseFence  = 0 ;
	Tbl[0].UseSelect = 0 ;

	//	Additional tools could fill up Tbl[1], Tbl[2], ...

	return (1) ;
}

//	=================================================================
//	Function for linking to TerraScan variables.
//
//	TerraScan calls this:
//	 - at startup (Flag == 0)
//	 - when point count has changed (Flag & AFLG_POINTCOUNT)
//	 - when point classes have been modified (Flag & AFLG_CLASSES)
//	 - when a new design file has been opened (Flag & AFLG_DESIGN)
//
//	Addon should store pointer and reflect modifications in
//	open windows if any.
//	=================================================================

void	AddonLinkVariables( AddLnk *Lp, int Flag)
{
	HWND	Drop ;
	int		Class ;

	//	Store pointer to variables

	LnkPtr = Lp ;

	//	Point count changed...
	
	if (Flag & AFLG_POINTCOUNT) {
		if (HisDlg)
			HisRecompute( HisDlg, 1) ;
	}

	//	Classes modified...

	if (Flag & AFLG_CLASSES) {
		if (HisDlg) {

			//	Refill class drop down list

			Drop = GetDlgItem( HisDlg, ID_HISCLASS);
			if (Drop)
				ClassFillDrop( Drop, LnkPtr->PtcTbl, LnkPtr->PtcCnt, HisClass) ;
			Class = ClassGetDropValue( Drop, LnkPtr->PtcTbl, LnkPtr->PtcCnt) ;
			if (Class != HisClass) {
				HisClass = Class ;
				HisRecompute( HisDlg, 1) ;
			}
		}
	}

	//	New coordinate system...

	if (Flag & AFLG_DESIGN) {
		//	No need to update
	}
}

//	=================================================================
//	Command invocation function. User has selected one of addon
//	modules menu commands.
//
//	Cmd is command number.
//   - Cmd is positive (0,1,2,...) if command was started from Addon
//     pulldown menu. This type of commands may open a dialog and
//	   stop for the user to select settings values.
//   - Cmd is -1, if command was started as Addon action in a macro.
//     Sp->Unparsed contains the information entered in the macro
//	   and may be parsed to determine actual routine to run and 
//	   what parameters it should use.
//	
//	Sp points to current state information.
//
//	Return value is a bit mask of ADDRET_xxxxx.
//	=================================================================

int		AddonRunCommand( int Cmd, AddSet *Sp)
{
	int		Ret = 0 ;
	
	//	Interactive menu commands

	if (Cmd == 0)
		Ret = ViewHistogramCmd( Sp) ;

	//	Macro commands for batch processing

	if (Cmd == -1)
		Ret = RunMacroStep( Sp->Cmd, Sp->Unparsed, Sp->BlkFile, Sp->ActFile) ;

	return (Ret) ;
}

//	=================================================================
//	Called before TerraScan is unloaded.
//
//	Addon module can release resources it has allocated.
//	=================================================================

void	AddonEnd( void)
{
	if (HisDlg)
		DestroyWindow(HisDlg) ;
}

//	=================================================================
//	Execute 'View histogram' command.
//
//	Return 0 always as we do not modify laser points.
//	=================================================================

int		ViewHistogramCmd( AddSet *Sp)
{
	HWND	Win ;

	Win = Sp->MainWin ;
	if (!HisDlg)
		HisDlg = CreateDialog( DllInst, "INTHIST", Win, HistogramDlg);
	if (HisDlg) {
		ShowWindow( HisDlg, SW_SHOWNORMAL) ;
		UpdateWindow( HisDlg) ;
	}
	return (0) ;
}

//	=================================================================
//	Dialog procedure for 'View histogram' dialog.
//	=================================================================

WINDLG	HistogramDlg( HWND Dlg, UINT Msg, WPARAM Par1, LPARAM Par2)
{
	BYTE			Sct[256] ;
	DRAWITEMSTRUCT	*Dp ;
	HWND			Drop ;
	HWND			Item ;
	RECT			Rt ;
	int				Idv ;
	int				Cmd ;

	if (Msg == WM_INITDIALOG) {

		//	Fill combo box with class names
		
		Drop = GetDlgItem( Dlg, ID_HISCLASS);
		if (Drop) {
			ClassFillDrop( Drop, LnkPtr->PtcTbl, LnkPtr->PtcCnt, HisClass) ;
			SetFocus( Drop) ;
		}

		//	Fill combo box with scanner numbers
		
		Drop = GetDlgItem( Dlg, ID_HISSCANNER);
		if (Drop) {
			ScannerGetUsed( Sct, *LnkPtr->ScrTbl, *LnkPtr->PntCnt) ;
			ScannerFillDrop( Drop, Sct, HisScanner) ;
			SetFocus( Drop) ;
		}

		//	Get histogram size and compute

		Item = GetDlgItem( Dlg, ID_HISPICTURE) ;
		if (Item) {
			GetWindowRect( Item, &Rt) ;
			HisHgt = Rt.bottom - Rt.top - (2 * HisMargY) ;
			HisTbl = GlobalAlloc( GMEM_FIXED, 65536 * sizeof(int)) ;
			HisRecompute( Dlg, 0) ;
		}
		return (1) ;
	}
	if (Msg == WM_DRAWITEM) {
		Dp = (DRAWITEMSTRUCT *) Par2 ;
		if (Dp->CtlID == ID_HISPICTURE)
			HisDrawPicture( Dp->hDC, &Dp->rcItem) ;			
	}
/*	if (Msg == WM_PAINT) {
		PAINTSTRUCT	Ps ;
		HDC			Dc ;

		Dc = BeginPaint( Dlg, &Ps) ;
		if (Dc) {
			Item = GetDlgItem( Dlg, ID_HISPICTURE) ;
			GetWindowRect( Item, &Rt) ;
			HisDrawPicture( Dc, &Rt) ;
		}
		EndPaint( Dlg, &Ps) ;
	}
*/	if (Msg == WM_COMMAND) {
		Idv = GET_WM_COMMAND_ID(Par1,Par2) ;
		Cmd = GET_WM_COMMAND_CMD(Par1,Par2) ;
		if ((Idv == ID_HISCLASS) || (Idv == ID_HISSCANNER)) {
			if (Cmd == CBN_SELCHANGE) {
				Item       = GetDlgItem( Dlg, ID_HISCLASS) ;
				HisClass   = ClassGetDropValue( Item, LnkPtr->PtcTbl, LnkPtr->PtcCnt) ; 
				Item       = GetDlgItem( Dlg, ID_HISSCANNER) ;
				HisScanner = ScannerGetDropValue( Item, *LnkPtr->ScrTbl) ;
				HisRecompute( Dlg, 1) ;
			}
		}
	}
	if (Msg == WM_CLOSE) {
		DestroyWindow( Dlg) ;
		return (1) ;
	}
	if (Msg == WM_DESTROY) {
		HisDlg = NULL ;
		if (HisTbl) {
			GlobalFree( HisTbl) ;
			HisTbl = NULL ;
		}
	}
	return (0) ;
}

//	=================================================================
//	Draw histogram picture into rectangle Rt.
//	=================================================================

void	HisDrawPicture( HDC Hdc, RECT *Rt)
{
	HBRUSH	Hbr ;
	HPEN	Red ;
	HPEN	Old ;
	RECT	Tb ;			//	Label area
	double	Mul = 0.0 ;
	double	Len ;
	double	Yml ;
	int		Stp ;
	int		Ys ;
	int		Xs ;
	int		Ox ;
	int		Oy ;
	int		Xp ;
	int		Yp ;
	int		K ;
	int		Y ;

	Hbr = GetSysColorBrush( GetSysColor(COLOR_BTNFACE)) ; 
	Rectangle( Hdc, Rt->left, Rt->top, Rt->right, Rt->bottom) ;

	Ys  = Rt->bottom - Rt->top - (2 * HisMargY) ;
	Xs  = Rt->right - Rt->left - (2 * HisMargX) ;
	Oy  = Rt->bottom - HisMargY ;
	Ox  = Rt->left + HisMargX + (Xs / 5) ;
	Xs -= (Xs / 5) + 4 ;	
	if (HisPeak > 0)
		Mul = (double) Xs / (double) HisPeak ;

	Yml = 0.0 ;
	if (HisMax)
		Yml = (double) Ys / (double) HisMax ;
	for( K=0 ; K < 65536 ; K++) {
		if (!HisTbl[K])				continue ;

		Y   = (int) ((K * Yml) + 0.5) ;
		Y   = max( Y, 0) ;
		Y   = min( Y, Ys) ;
		Len = Mul * HisTbl[K] ;
		Yp  = Oy - Y ;
		Xp  = Ox + ((int) (Len + 0.5)) ;
		if (Xp == Ox)
			Xp++ ;
		MoveToEx( Hdc, Ox, Yp, NULL) ;
		LineTo( Hdc, Xp, Yp) ;
	}

	//	Draw axis

	Red = CreatePen( PS_SOLID, 1, RGB(255,0,0)) ; 
	Old = SelectObject( Hdc, Red) ;
	MoveToEx( Hdc, Ox-1, Oy, NULL) ;
	LineTo( Hdc, Ox+Xs, Oy) ;
	MoveToEx( Hdc, Ox-1, Oy, NULL) ;
	LineTo( Hdc, Ox-1, Oy-Ys) ;

	//	Label minimum and maximum

	Tb.left   = Rt->left + 1 ;
	Tb.right  = Ox - 1 ;
	Tb.top    = Oy - Ys ;
	Tb.bottom = Oy ;
	SetTextColor( Hdc, RGB(255,0,0)) ;
	SetBkMode( Hdc, TRANSPARENT) ;
	HisDrawLabel( Hdc, 0, HisMax, &Tb) ;
	HisDrawLabel( Hdc, HisMax, HisMax, &Tb) ;

	//	Label intermediate values

	Stp = HisLabelStep(HisMax) ;
	for( Y=Stp ; Y < HisMax - (Stp/2) ; Y += Stp)
		HisDrawLabel( Hdc, Y, HisMax, &Tb) ;

	//	Release created pen

	SelectObject( Hdc, Old) ;
	DeleteObject( Red) ;
}

//	=================================================================
//	Draw label for intensity value Val.
//	Rectangle Tb is area reserved for labels.
//	Max is maximum intensity value.
//	=================================================================

void	HisDrawLabel( HDC Hdc, int Val, int Max, RECT *Tb)
{
	UINT	Fmt = DT_RIGHT | DT_VCENTER | DT_SINGLELINE ;
	char	Buf[200] ;
	RECT	Rt ;
	double	Rat ;
	int		Cy ;

	if (!Max)									return ;

	wsprintf( Buf, "%d", Val) ;
	Rat = (double) Val / (double) Max ;
	Cy  = Tb->bottom - (int) (Rat * (Tb->bottom - Tb->top) + 0.5) ;

	Rt.top    = Cy - 8 ;
	Rt.bottom = Cy + 8 ;
	Rt.left   = Tb->left + 1 ;
	Rt.right  = Tb->right - 8 ;
	DrawText( Hdc, Buf, strlen(Buf), &Rt, Fmt) ;

	//	Draw tick line

	MoveToEx( Hdc, Tb->right, Cy, NULL) ;
	LineTo( Hdc, Tb->right-2, Cy) ;
}

//	=================================================================
//	Return labeling interval when range is from zero to Max.
//	=================================================================

int		HisLabelStep( int Max)
{
	if (Max > 40000)					return (10000) ;
	if (Max > 20000)					return (5000) ;
	if (Max > 10000)					return (2000) ;
	if (Max > 4000)						return (1000) ;
	if (Max > 2000)						return (500) ;
	if (Max > 1000)						return (200) ;
	if (Max > 400)						return (100) ;
	if (Max > 200)						return (50) ;
	if (Max > 100)						return (20) ;
	if (Max > 40)						return (10) ;
	if (Max > 20)						return (5) ;
	if (Max > 10)						return (2) ;

	return (1) ;
}

//	=================================================================
//	Recompute histogram values and optionally redraw picture.
//	=================================================================

void	HisRecompute( HWND Dlg, int Redraw)
{
	HCURSOR	Cursor ;
	char	Ts[400] ;
	HWND	Item ;
	USHORT	*Int ;
	BYTE	*Mrk ;
	BYTE	*Cls ;
	BYTE	*Scr ;
	int		Cnt ;

	Cursor = SetCursor( LoadCursor( NULL, IDC_WAIT)) ;
	Int    = *LnkPtr->IntTbl ;
	Mrk    = *LnkPtr->MrkTbl ;
	Cls    = *LnkPtr->ClsTbl ;
	Scr    = *LnkPtr->ScrTbl ;
	Cnt    = *LnkPtr->PntCnt ;

	HisMax   = HisCompute( &HisPeak, HisTbl, Int, Mrk, Cls, Scr, Cnt, HisClass, HisScanner) ;
	if (Redraw) {
		Item = GetDlgItem( Dlg, ID_HISPICTURE) ;
		if (Item)
			InvalidateRect( Item, NULL, 1) ;
	}

	//	Compute average and median values

	HisCompAveMedian( &HisAve, &HisMedian, &HisSpread, HisTbl, HisMax+1) ;
	sprintf( Ts, "Average %.1f", HisAve) ;
	SetWindowText( GetDlgItem( Dlg, ID_HISAVERAGE), Ts) ;
	sprintf( Ts, "Median %.0f", HisMedian) ;
	SetWindowText( GetDlgItem( Dlg, ID_HISMEDIAN), Ts) ;
	sprintf( Ts, "Spread %.0f", HisSpread) ;
	SetWindowText( GetDlgItem( Dlg, ID_HISSPREAD), Ts) ;

	SetCursor( Cursor) ;
}

//	=================================================================
//	Compute histogram values from laser points.
//
//	If Peak != NULL, find maximum count in Tbl[].
//
//	Return maximum intensity value.
//	=================================================================

int		HisCompute( int *Peak, int *Tbl, USHORT *Int, BYTE *Mrk, BYTE *Cls, BYTE *Scr, int Pct, int Class, int Scv)
{
	USHORT	Imn ;
	USHORT	Imx ;
	int		Mark = 1 ;
	int		Cnt ;
	int		Use ;
	int		Ind ;
	int		Mx ;
	int		K ;

	//	Initialize counts to zero

	for( K=0 ; K < 65536 ; K++)
		Tbl[K] = 0 ;

	//	Mark points to use

	ScanMarkAll( Mrk, Pct, 0) ;
	Use = ScanMarkClass( Mrk, Cls, Pct, Class, 1) ;
	if (!Use)									return (0) ;

	//	Mark by scanner

	if ((Scr) && (Scv != SCANNER_ANY)) {
		Mark = 2 ;
		ScanMarkScanner( Mrk, Scr, Pct, Scv, 1, 2) ;
	}

	//	Find minimum and maximum intensity

	ScanIntensityRange( &Imn, &Imx, Int, Mrk, Pct, Mark) ;

	//	Add to counts

	for( K=0 ; K < Pct ; K++) {
		if (Mrk[K] != Mark)			continue ;
		Ind = Int[K] ;
		Tbl[Ind]++ ;
	}

	//	Find maximum count value

	if (Peak) {
		Mx  = Tbl[0] ;
		Cnt = Imx + 1 ; 
		for( K=1 ; K < Cnt ; K++)
			Mx = max( Mx, Tbl[K]) ;
		*Peak = Mx ;
	}
	return (Imx) ;
}

//	=================================================================
//	Compute statistical values from histogram:
//	  *Ave   average intensity
//	  *Med   median intensity
//	  *Spr   spread (80% points are between median-spread and
//	         median+spread
//
//	Return 1 on success.
//	Return 0 if failed (histogram is empty).
//	=================================================================

int		HisCompAveMedian( double *Ave, double *Med, double *Spr, int *Tbl, int Cnt)
{
	double	Sum = 0.0 ;
	double	Div = 0.0 ;
	double	Lim ;
	int		Ind ;
	int		Mdv ;
	int		Spv ;
	int		K ;

	//	Initialize to zero

	if (Ave)
		*Ave = 0.0 ;
	if (Med)
		*Med = 0.0 ;

	//	Collect sum of intensities

	for( K=0 ; K < Cnt ; K++) {
		if (!Tbl[K])			continue ;

		Sum += (double) Tbl[K] * (double) K ;
		Div += (double) Tbl[K] ;
	}
	if (!Div)							return (0) ;

	//	Compute average

	if (Ave)
		*Ave = Sum / Div ;

	//	Determine median

	if (Med) {
		Mdv = 32768 ;
		Sum = 0.0 ;
		Lim = 0.5 * Div ;
		for( K=0 ; K < Cnt ; K++) {
			if (!Tbl[K])			continue ;

			Sum += Tbl[K] ;
			if (Sum >= Lim) {
				Mdv  = K ;
				*Med = K ;
				break ;
			}
		}
		
		//	Determine spread where 80% points fall

		Spv = 0 ;
		Lim = 0.8 * Div ;
		Sum = Tbl[Mdv] ;
		while (Sum < Lim) {
			Spv++ ;
			Ind = Mdv - Spv ;
			if (Ind > 0)
				Sum += Tbl[Ind] ;
			Ind = Mdv + Spv ;
			if (Ind < 65536)
				Sum += Tbl[Ind] ;
		}
		*Spr = Spv ;
	}
	return (1) ;
}

//	=================================================================
//	Fill drop down list Drop with "Any class" selection plus class
//	names.
//
//	S points to active state information.
//	SelVal specified class to selected.
//	=================================================================

void	ClassFillDrop( HWND Drop, PntCls *Tbl, int Cnt, int SelVal)
{
	char	Buf[400] ;
	PntCls	*Cp ;
	int		SelInd = 0 ;
	int		K ;

	SendMessage( Drop, WM_SETREDRAW, 0, 0);
	SendMessage( Drop, CB_RESETCONTENT, 0, 0) ;

	SendMessage( Drop, CB_ADDSTRING, 0, (LPARAM) "Any class") ;
	for( K=0 ; K < Cnt ; K++) {
		Cp = Tbl + K ;
		wsprintf( Buf, "%d %s", Cp->Class, Cp->Desc) ;
		SendMessage( Drop, CB_ADDSTRING, 0, (LPARAM) Buf) ;
		if (Cp->Class == SelVal)
			SelInd = K + 1 ;
	}
	SendMessage( Drop, CB_SETCURSEL, SelInd, 0);
	SendMessage( Drop, WM_SETREDRAW, 1, 0);
	InvalidateRect( Drop, NULL, 1);
}

//	=================================================================
//	Get currently selected class value from drop down list.
//	=================================================================

int		ClassGetDropValue( HWND Drop, PntCls *Tbl, int Cnt)
{
	int		Ind ;

	//	Validate parameters

	if ((!Drop) || (!Tbl))			return (CLASS_ANY) ;

	//	Get selected row

	Ind = SendMessage( Drop, CB_GETCURSEL, 0, 0) ;
	if (Ind <= 0)					return (CLASS_ANY) ;
	if (Ind > Cnt)					return (CLASS_ANY) ;

	return (Tbl[Ind-1].Class) ;
}

//	=================================================================
//	Fill table Sct[256] to indicate what scanner numbers are used.
//
//	Set Sct[x] == 1 for every scanner number which apppears in Scr[].
//	Set Sct[x] == 0 for every scanner number not used.
//
//	Return number of different scanner numbers found.
//	=================================================================

int		ScannerGetUsed( BYTE *Sct, BYTE *Scr, int Cnt)
{
	int		Ret = 0 ;
	int		K ;
	int		I ;

	//	Validate

	if (!Sct)							return (0) ;

	//	Initialize to zero

	for( K=0 ; K < 256 ; K++)
		Sct[K] = 0 ;
	if (!Scr)							return (0) ;
	if (Cnt <= 0)						return (0) ;

	//	Loop thru laser point scanner numbers

	for( K=0 ; K < Cnt ; K++) {
		I = Scr[K] ;
		if (!Sct[I]) {
			Ret++ ;
			Sct[I] = 1 ;
		}
	}
	return (Ret) ;
}

//	=================================================================
//	Fill drop down list Drop with "Any scanner" selection plus used
//	scanner numbers "1", "2", ...
//
//	SelVal specified scanner to select.
//	=================================================================

void	ScannerFillDrop( HWND Drop, BYTE *Sct, int SelVal)
{
	char	Buf[400] ;
	int		SelInd = 0 ;
	int		Ind = 0 ;
	int		K ;

	SendMessage( Drop, WM_SETREDRAW, 0, 0);
	SendMessage( Drop, CB_RESETCONTENT, 0, 0) ;

	SendMessage( Drop, CB_ADDSTRING, 0, (LPARAM) "Any scanner") ;

	for( K=0 ; K < 256 ; K++) {
		if (!Sct[K])				continue ;

		wsprintf( Buf, "%d", K) ;
		SendMessage( Drop, CB_ADDSTRING, 0, (LPARAM) Buf) ;
		Ind++ ;
		if (SelVal == K)
			SelInd = Ind ;
	}
	SendMessage( Drop, CB_SETCURSEL, SelInd, 0);
	SendMessage( Drop, WM_SETREDRAW, 1, 0);
	InvalidateRect( Drop, NULL, 1);
}

//	=================================================================
//	Get currently selected scanner value from drop down list.
//	=================================================================

int		ScannerGetDropValue( HWND Drop, BYTE *Scr)
{
	char	Buf[400] ;
	int		Nbr ;
	int		Ind ;

	//	Validate parameters

	if (!Drop)						return (SCANNER_ANY) ;
	if (!Scr)						return (SCANNER_ANY) ;

	//	Get selected row

	Ind = SendMessage( Drop, CB_GETCURSEL, 0, 0) ;
	if (Ind <= 0)					return (SCANNER_ANY) ;

	SendMessage( Drop, WM_GETTEXT, 400, (LPARAM) Buf) ;
	if (Buf[0] < '0')				return (SCANNER_ANY) ;
	if (Buf[0] > '9')				return (SCANNER_ANY) ;

	Nbr = atoi(Buf) ;
	if (Nbr < 0)					return (SCANNER_ANY) ;
	if (Nbr > 255)					return (SCANNER_ANY) ;

	return (Nbr) ;
}

//	=================================================================
//	Run addon macro step.
//
//	Cmd is command name.
//	Unparsed points additional parameters as typed into macro.
//	BlkFile contains full path to active block file.
//	ActFile contains full path to active file.
//
//	Return value >= 0 indicates successful completion.
//	Return value < 0 indicates failure.
//	=================================================================

int		RunMacroStep( char *Cmd, char *Unparsed, char *BlkFile, char *ActFile)
{
	Point3d	*Pnt ;		//	Pointer to point coordinates
	double	*Dbl ;		//	Pointer to double time stamps
	double	Val ;
	USHORT	*Int ;		//	Pointer to intensity values
	BYTE	*Ech ;		//	Pointer to echo bits
	BYTE	*Mrk ;		//	Pointer to mark values
	int		Cnt ;

	//	Validate state

	if (!LnkPtr)								return (-2) ;

	//	Get pointers to TerraScan table of points

	Pnt = *LnkPtr->PntTbl ;
	Dbl = *LnkPtr->DblTbl ;
	Int = *LnkPtr->IntTbl ;
	Ech = *LnkPtr->EchTbl ;
	Mrk = *LnkPtr->MrkTbl ;
	Cnt = *LnkPtr->PntCnt ;

	//	AssignEcho (needs no parameters)

	if (!lstrcmpi( Cmd, "AssignEcho"))
		return (ScanAssignEcho( Ech, Pnt, Mrk, Dbl, Cnt)) ;

	//	MultiplyIntensity x.xx
	//	  where x.xx is scale factor for intensities

	if (!lstrcmpi( Cmd, "MultiplyIntensity")) {
		Val = atof(Unparsed) ;
		return (ScanMultiplyIntensity( Int, Cnt, Val)) ;
	}
	return (-1) ;
}

//	=================================================================
//	Assign echo information for laser points. Routine requires that:
//	- double precision time stamps are present (LAS files)
//	- laser points have been sorted into time order
//
//	Routine looks at how many consecutive laser points have the same
//	time stamp. It assigns echo information based on elevation of
//	the points.
//
//	Routine changes:
//	 - Tbl[x].Echo field
//
//	Return 1 on success.
//	Return 0 if nothing was done.
//	=================================================================

int		ScanAssignEcho( BYTE *Ech, Point3d *Pnt, BYTE *Mrk, double *Dbl, int Cnt)
{
	int		EchoCnt ;
	int		Echo ;
	int		Msk ;
	int		Clr ;
	int		Hi ;
	int		Hz ;
	int		Pz ;
	int		A ;
	int		I ;
	int		K ;

	//	We must have points and time stamps

	if (!Ech)							return (0) ;
	if (!Pnt)							return (0) ;
	if (!Mrk)							return (0) ;
	if (!Dbl)							return (0) ;
	if (Cnt <= 0)						return (0) ;

	//	Mark all points as not processed (Mark=0)

	ScanMarkAll( Mrk, Cnt, 0) ;

	//	Loop thru points

	K = 0 ;
	while (K < Cnt) {

		//	Find how many points have the same time stamp

		for( A=K+1 ; A < Cnt ; A++)
			if (Dbl[A] != Dbl[K])	break ;
		EchoCnt = A - K ;

		//	Loop to assign echo values

		Echo = 1 ;
		while (Echo <= EchoCnt) {

			//	Find highest echo

			Hi   = -1 ;
			Hz   = 0 ;
			for( I=K ; I < A ; I++) {
				if (Mrk[I])				continue ;	
				Pz = Pnt[I].z ;
				if (Hi >= 0)
					if (Pz <= Hz)		continue ;
				Hz = Pz ;
				Hi = I ;
			}
			if (Hi < 0)					break ;

			//	Assign echo number and echo count to LAS bits

			Msk = Echo | (EchoCnt << 3) ;
			Clr = Ech[Hi] & 0xC0 ;
			Ech[Hi] = Clr | Msk ;
			Mrk[Hi] = 1 ;		//	Point processed

			//	Increase echo number

			Echo++ ;
		}

		//	Increase loop counter

		K += EchoCnt ;
	}
	return (1) ;
}

//	=================================================================
//	Multiply intensity values with Mul.
//
//	Return 1 on success.
//	Return 0 if nothing was done.
//	=================================================================

int		ScanMultiplyIntensity( USHORT *Int, int Cnt, double Mul)
{
	double	Val ;
	int		K ;

	//	We must have intensity values

	if (!Int)							return (0) ;
	if (Cnt <= 0)						return (0) ;

	//	Loop thru points

	for( K=0 ; K < Cnt ; K++) {
		Val = Int[K] * Mul ;
		Int[K] = (USHORT) (Val + 0.5) ;
	}
	return (1) ;	
}

//	=================================================================
//	Mark all laser points in Tbl[] with mark value Mark.
//	=================================================================

void	ScanMarkAll( BYTE *Mrk, int Cnt, int Mark)
{
	int		K ;

	for( K=0 ; K < Cnt ; K++)
		Mrk[K] = Mark ;
}

//	=================================================================
//	Mark all laser points in class Class with mark value Mark.
//
//	Return number of points marked.
//	=================================================================

int		ScanMarkClass( BYTE *Mrk, BYTE *Cls, int Cnt, int Class, int Mark)
{
	int		Ret = 0 ;
	int		K ;

	//	CLASS_ANY marks all points

	if (Class == CLASS_ANY) {
		ScanMarkAll( Mrk, Cnt, Mark) ;
		return (Cnt) ;
	}

	//	Mark points only in given class

	for( K=0 ; K < Cnt ; K++) {
		if (Cls[K] != Class)		continue ;
		Ret++ ;
		Mrk[K] = Mark ;
	}
	return (Ret) ;
}

//	=================================================================
//	Change mark value of laser points from scanner Scv.
//
//	Return number of points marked.
//	=================================================================

int		ScanMarkScanner( BYTE *Mrk, BYTE *Scr, int Cnt, int Scv, int From, int To)
{
	int		Ret = 0 ;
	int		K ;

	//	Mark points only from given scanner

	for( K=0 ; K < Cnt ; K++) {
		if (Mrk[K] != From)				continue ;
		if (Scv != SCANNER_ANY)
			if (Scr[K] != Scv)			continue ;
		Ret++ ;
		Mrk[K] = To ;
	}
	return (Ret) ;
}

//	=================================================================
//	Find minimum and maximum intensity among marked laser points.
//
//	Return 1 if a valid range was determined.
//	Return 0 if no marked points.
//	=================================================================

int		ScanIntensityRange( USHORT *Mn, USHORT *Mx, USHORT *Int, BYTE *Mrk, int Cnt, int Mark)
{
	USHORT	Vmn ;
	USHORT	Vmx ;
	int		Ret = 0 ;
	int		K ;

	for( K=0 ; K < Cnt ; K++) {
		if (Mrk[K] != Mark)			continue ;
		if (Ret) {
			Vmn = min( Vmn, Int[K]) ;
			Vmx = max( Vmx, Int[K]) ;
		}
		else {
			Ret = 1 ;
			Vmn = Int[K] ;
			Vmx = Int[K] ;
		}
	}
	if (Ret) {
		*Mn = Vmn ;
		*Mx = Vmx ;
	}
	return (Ret) ;
}

//	=================================================================
//	Convert master unit point D to internal unit point P.
//
//	This converts meters (master units) to laser point coordinates.
//	=================================================================

void	DpToIp( Point3d *P, Dp3d *D, AddLnk *Lp)
{
	P->x = DblToLong( D->x * Lp->Units + Lp->OrgX) ;
	P->y = DblToLong( D->y * Lp->Units + Lp->OrgY) ;
	P->z = DblToLong( D->z * Lp->Units + Lp->OrgZ) ;
}

//	=================================================================
//	Convert internal unit point P to master unit point D.
//
//	This converts laser point coordinates to meters (master units).
//	=================================================================

void	IpToDp( Dp3d *D, Point3d *P, AddLnk *Lp)
{
	D->x = (P->x - Lp->OrgX) * Lp->Units ;
	D->y = (P->y - Lp->OrgY) * Lp->Units ;
	D->z = (P->z - Lp->OrgZ) * Lp->Units ;
}

//	=================================================================
//	Convert double to long value.
//	=================================================================

long	DblToLong( double D)
{
	if (D < 0.0)
		return ((long) (D - 0.5)) ;

	return ((long) (D + 0.5)) ;
}
